/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef RF_SYSTEM_1CHANNEL_ODC_H
#define RF_SYSTEM_1CHANNEL_ODC_H
#include "rf_system.h"
#include "rcm_class.h"
#include "gpioclassv2.h"
#include <QTimer>
#include <QThread>
#include <QProcess>

#ifndef ALT_MODBUS_IMPLEMENTATION
#include "modbusserver.h"
#else
#include "modbusserver_v2.h"
#endif

#ifdef ETHERCAT
#include "ethercat/signalsslots.h"
#endif

#define CHANNELCOUNT 1

#ifdef I2C_TEMP_SENSOR
/**********************************************************************************************************************************************************************************
 * i2C temp sensor class
 *********************************************************************************************************************************************************************************/
class i2cSensor : public QObject
{
	Q_OBJECT

public:
	i2cSensor();
	~i2cSensor();

private:
    QProcess *sensor_process;
	QTimer *sensor_timer;		//Timer to poll the all_Ok signal on the GPIO;

public slots:
    void Initialize();
	void Execute_Script();
	void Parse_Data();

signals:
	void signal_temperature_sensor_get(double val);
};
#endif
/**********************************************************************************************************************************************************************************
 * i2C temp sensor class
 *********************************************************************************************************************************************************************************/
class RF_System_1channel_ODC : public QObject
{
	Q_OBJECT

public:
	RF_System_1channel_ODC();
	~RF_System_1channel_ODC();

	QList<QThread*> *threadList;
	RCM_Class *RCM_USB_port;
	ConfigHandler *config;
	ModbusServer *mbServer;
	
#ifdef ETHERCAT
	EtherCATSignalsSlots *ethercatSs;
#endif

#ifdef I2C_TEMP_SENSOR
	i2cSensor *sensor;
#endif

	int channelCount();

private:
//	void firmware_version_check();
//	int	firmware_version_actual[4];
//	int	firmware_version_requirement[4];

	QThread *channelThread;
	QThread *modbusThread;
	QThread *sensorThread;
	RF_Channel *channel;
	int channel_count = CHANNELCOUNT;

	bool channels_working = true;
	bool PSUs_readings_valid = true;
	bool last_interlock_state = false;			//track if last OK was OK.
	bool PSU_interlock_modbus = false;
	bool get_interlock_state();
	QTimer *ready_check_timer;		//Timer to poll the all_Ok signal on the GPIO;
	QTimer *PSU_timer;			//Introduce a non-blocking delay for the PSU_enable on OK signal; PSU's may still be in their booting phase and ignore the

	GPIOClass *GPIO_interlock;

public slots:
	//ODC WORKAROUND
	void execute_sweep_dBm_workaround(int intrasys_num, double freq_start, double freq_stop, double freq_step, double pow_dbm);
	void set_RF_enable_workaround(int intra_sys_num, bool enable);

	void PSU_timer_action();
	void update_PSU(bool enable);
	void handler_PSU_interlock_modbus(bool enable);
	void handler_channel_working(int channel_number, bool enabled);
	void handler_PSU_readings_valid(int channel_number, bool valid);
	void handler_reset_detected(int intrasys_num, int channel_number);
	void poll_generator_ready();

	void RCM_USB_message_handler();
	void RCM_USB_response_handler(QString response);
	void RCM_USB_message_handler_blind();
	void RCM_USB_response_handler_blind(QString response);
	void RCM_USB_error_handler(QSerialPort::SerialPortError error);

signals:
	void signal_RCM_USB_message(QString message);
	void signal_RCM_sweep_message(QString message);
	void signal_interlock_enable(bool ok);
	void signal_set_generator_ready(bool state);
	void signal_SGx_communication_working(bool state);
	void signal_PSU_communication_working(bool state);
	void signal_set_RF_enable(int intrasys_num, bool enable);
	void signal_set_PSU_enable(int intrasys_num, bool enable);
	void signal_set_PSU_voltage_setpoint(int intrasys_num, double voltage);
	void signal_set_PSU_current_limit(int intrasys_num, double current_limit);

	//ODC workaround signals
	void signal_set_power_control_mode(int intrasys_num, int mode);
	void signal_execute_sweep_2(int intrasys_num, double freq_start, double freq_stop, double freq_step, double pow_dbm);
//	void signal_set_RF_enable();
};

#endif // RF_SYSTEM_1CHANNEL_ODC_H
